# Bringing HLSL shaders to Metal

Convert ray query HLSL shaders to Metal IR using the Metal shader converter.

## Overview

This sample code project shows how to bring HLSL shaders to Metal by converting them into 
Metal IR (intermediate representation). The sample app first compiles its HLSL shaders to DXIL 
(DirectX intermediate language), placing the results in the application bundle. At runtime, the 
sample app converts these DXIL files to Metal libraries, uses the libraries 
to create pipeline state instances, and then renders a scene.

The sample code project demonstrates a compute shader that performs ray query.

- Note: This sample code project is associated with WWDC23 session 8181: [Bring your game to Mac: Compile your shaders](https://developer.apple.com/wwdc23/8181).

## Configure the sample code project

Before you run the sample code project in Xcode, ensure that you're using macOS 13, or later, and Xcode 15.

This project depends on the **Metal Shader Converter**. It searches for the header files under `/usr/local/include/` and the dynamic library under `/usr/local/lib/`. Make sure you install the **Metal Shader Converter** files in these locations.

This project includes pre-compiled versions of its HLSL files. In order to modify them, you need to build the **DirectX Shader Compiler**.

## Compile DXC

The sample project recompiles the HLSL files when it finds the `dxc` command-line tool.

Follow these steps to clone and build `dxc` on your device:

```bash
git clone https://github.com/microsoft/DirectXShaderCompiler.git
cd DirectXShaderCompiler
mkdir build
cd build
cmake .. -G "Unix Makefiles" -DCMAKE_BUILD_TYPE=Release -C ../cmake/caches/PredefinedParams.cmake -DCMAKE_OSX_ARCHITECTURES="x86_64;arm64"
make -j8
```

After the build process completes, copy the files `dxc-3.7` and `libdxcompiler-3.7.dylib` to `/usr/local/bin/` and `/usr/local/lib` respectively, and create a symlink to `libdxcompiler-3.7.dylib` named `libdxcompiler.dylib`.

```bash
sudo ditto bin/dxc-3.7 /usr/local/bin/
sudo ditto lib/libdxcompiler.dylib /usr/local/lib/
sudo ln -s /usr/local/lib/libdxcompiler.dylib /usr/local/lib/libdxcompiler-3.7.dylib
```

## Compile HLSL to DXIL

Because the sample code project uses HLSL as its shader format, if the `dxc` command-line tool is available, the "Compile HLSL" build phase invokes it with its appropriate arguments.

The code sample's Xcode project specifies the following:

```bash
DXC=/usr/local/bin/dxc-3.7
if [ -f "${DXC}" ]; then
  "${DXC}" -T cs_6_6 -Fo "${SCRIPT_OUTPUT_FILE_0}" "${SCRIPT_INPUT_FILE_0}" -Zi -Qembed_debug
fi
```

Where the parameters denote:

* `-T cs_6_6` - the shader type and version
* `-Fo "${SCRIPT_OUTPUT_FILE_0}"` - the output DXIL file
* `"${SCRIPT_INPUT_FILE_0}"` - the input HLSL file
* `-Zi -Qembed_debug` – enable debug information

To make these DXIL files available at runtime, a subsequent "Copy Files" phase copies the output DXIL files into the application bundle.

- Note: If the sample cannot find the `dxc` command-line tool it uses prebuilt DXIL files.

## Compile DXIL to Metal IR

To load its shader libraries, the sample uses `libmetalirconverter` to convert DXIL to Metal IR.

- Note: The sample converts DXIL to Metal IR at runtime to speed up your game's bringup process. When your game is ready for release, remove this step and adopt ahead-of-time conversion by using `metal-shaderconverter` during the build process.

The sample uses the functions in `metal_irconverter.h` to perform the conversion.

```cpp
IRCompiler* pCompiler = IRCompilerCreate();
IRCompilerSetEntryPointName(pCompiler, entryPointName);

IRError* pError = nullptr;
IRCompilerSetGlobalRootSignature(pCompiler, pRootSig);

IRObject* pAIR = IRCompilerAllocCompileAndLink(pCompiler, nullptr, pDXIL, &pError);
```

The code example above creates a compiler instance using `IRCompilerCreate`. It then sets the minimum deployment target for the Metal IR using `IRCompilerSetMinimumDeploymentTarget`.

- Note: Always set a deployment target equal to or lower than your SDK.

`IRCompilerSetEntryPointName` customizes the name of the entry point and `IRCompilerSetGlobalRootSignature` 
explicitly defines the layout of the top-level argument buffer. Then `IRCompilerAllocCompileAndLink` converts DXIL to Metal IR.

- Note: To compile DXIL to Metal IR at runtime, the sample copies the `libmetalirconverter.dylib` 
into the application bundle in a custom build phase.

## Use the converted shaders to create Metal libraries

To use the converted shaders, the sample creates `MTL::Library` instances from the Metal IR. `IRMetalLibGetBytecode` retrieves the bytecode corresponding to the Metal IR.

The app provides this bytecode to the `MTL::Device` to create an `MTL::Library` instance by copying it into a `dispatch_data_t` object. `pDevice->newLibrary` then creates the 
library instance, which the sample uses to create its pipeline state objects.

```cpp
uint8_t* metallibBytecode = new uint8_t[metallibSize];
IRMetalLibGetBytecode(pMetalLib, metallibBytecode);

dispatch_data_t metallib = dispatch_data_create(metallibBytecode, metallibSize, dispatch_get_main_queue(), DISPATCH_DATA_DESTRUCTOR_DEFAULT);
MTL::Library* pLib       = pDevice->newLibrary(metallib, nullptr);
```

## Create pipeline state objects

Metal apps use pipeline state objects to configure the GPU for rendering. 

This sample creates a compute pipeline state object at load time by converting the ray query shader to MetalIR, retrieving the converted shader function from the `MTL::Library` instance, and 
using it to create a `MTL::ComputePipelineState` instance via `pDevice->newComputePipelineState`.

```cpp
NS::Error* pError;
MTL::Function* pFn = pLib->newFunction(MTLSTR("MainCS"));
MTL::ComputePipelineState* pComputePSO =
                     pDevice->newComputePipelineState(pFn,
                     &pError);
```

## Encode argument buffers

To simplify common tasks, such as encoding argument buffers, the sample leverages the `metal_irconverter_runtime.h` header. It uses the `IRDescriptorTableEntry` structure to create aggregate data types, matching the layout of the argument buffers, and the helper functions such as `IRDescriptorTableSetTexture` to encode texture references.

```cpp
IRDescriptorTableSetTexture((IRDescriptorTableEntry*)pUAVTable->contents(), _pTexture, 0, 0);
```

## Raytrace the scene

This sample builds a Metal instance acceleration structure and binds it to the compute pipeline through an Acceleration Structure header that references the
instance contribution to the hit index. The sample uses the same header buffer to store the instance contributions to avoid loading from a second buffer, and 
for simplicity, but this is not a requirement.

```cpp
uint32_t instanceContributions[] = { 0, 0 };
NS::UInteger headerSize = sizeof(IRRaytracingAccelerationStructureGPUHeader) + sizeof(uint32_t) * NUM_ELEMS(instanceContributions);
MTL::Buffer* pAccelStructureHdrBuffer = _pDevice->newBuffer(headerSize, MTL::ResourceStorageModeShared)->autorelease();

IRRaytracingSetAccelerationStructure((uint8_t *)pAccelStructureHdrBuffer->contents(),
                                      _instanceAccelerationStructure->gpuResourceID(),
                                      (uint8_t *)pAccelStructureHdrBuffer->contents() + sizeof(IRRaytracingAccelerationStructureGPUHeader),
                                      instanceContributions, NUM_ELEMS(instanceContributions));
```

It then dispatches the compute workload that makes the compute kernel trace rays against the acceleration structure via ray query, and paints the pixels on an image accordingly.

```cpp
// Dispatch threads:
NS::UInteger threadGroupSize = _triangleSphereRTPipelineContext.pRTPSO->maxTotalThreadsPerThreadgroup();
MTL::Size threadgroupSize(threadGroupSize, 1, 1);

MTL::Size gridSize = MTL::Size(kTextureWidth, kTextureHeight, 1);
pComputeEncoder->dispatchThreads(gridSize, threadgroupSize);
```

The sample then presents this image to the user.

* Note: Because the compute kernel access the instance and primitive acceleration structures indirectly through this buffer, the sample marks them resident via the `useResources()` API.

## Using this sample to bring up your game

If you use this project as a starting template for your game, follow these steps to add your existing HLSL files to the Xcode project.

- Note to build your custom HLSL files you need to set up the `dxc` command-line tool.

1. Add the HLSL files to the project hierarchy.
2. Click the project's name at the top of the navigator.
3. Select **Build Phases**.
4. Open the **Compile HLSL** phase.
5. Under **Input Files**, add the relative path to your new HLSL file.
6. Under **Output Files**, add a new entry for each DXIL file the project generates. For example, add one entry for the vertex DXIL file and one for the fragment DXIL file.
7. Edit the shell script and add a build command for each new output file. Remember input and output files are zero-based.
8. Compile your project. If there are any mistakes in the HLSL or the build commands Xcode displays the reason in the build output.
9. After a successful build, open the "dxil" folder in the project view. Right click and select **add files**.
10. Navigate to the dxil folder in your filesystem. Locate the new files and add them to the project.
11. Close the **Compile HLSL** phase and open the **Copy Bundle Resources** phase.
12. Review the new files and (optionally) remove any unnecessary ones.
