//-------------------------------------------------------------------------------------------------------------------------------------------------------------
//
// Copyright 2023 Apple Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
//-------------------------------------------------------------------------------------------------------------------------------------------------------------

#ifndef SHADERCOMPILER_HPP
#define SHADERCOMPILER_HPP

#include <Metal/Metal.hpp>
#include <metal_irconverter/metal_irconverter.h>

#include <string>
#include <tuple>

/// This sample compiles DXIL to Metal IR at runtime to demonstrate the programmatic interface to IRConverter.
/// Although this is fast, you can improve app startup times by generating the metallibs as part of the build
/// process and then, at runtime, loading them directly into the Metal device.

/// Load and compile a DXIL file into a metallib, and return it.
/// The caller is responsible for releasing the library.
MTL::Library* newLibraryFromDXIL(const std::string& dxilPath,
                                 IRShaderStage shaderStage,
                                 const char* entryPointName,
                                 const IRRootSignature* pRootSig,
                                 MTL::Device* pDevice);


/// Load and compile a DXIL file into a metallib, and return it alongside reflection data.
/// The caller is responsible for releasing the library and shader reflection.
std::pair<MTL::Library*, IRShaderReflection*> newLibraryWithReflectionFromDXIL(
                                                                    const std::string& dxilPath,
                                                                    IRShaderStage shaderStage,
                                                                    const char* entryPointName,
                                                                    const IRRootSignature* pRootSignature,
                                                                    MTL::Device* pDevice,
                                                                    bool enableGSEmulation);


/// Generate a stage-in function for the input layout descriptor. IRConverter places the function
/// in its own metallib. The caller is responsible for releasing the returned library.
MTL::Library* newStageInLibrary(const IRVersionedInputLayoutDescriptor& inputLayoutDescriptor,
                                IRShaderReflection* pVertexStageReflection,
                                MTL::Device* pDevice);

#endif // SHADERCOMPILER_HPP
