//-------------------------------------------------------------------------------------------------------------------------------------------------------------
//
// Copyright 2025 Apple Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
//-------------------------------------------------------------------------------------------------------------------------------------------------------------


//-------------------------------------------------------------------------------------------------------------------------------------------------------------
// This project includes a pre-compiled version of this file.
// In order to modify it, you need to build the DirectX
// Shader Compiler. Please see the README for instructions.
//-------------------------------------------------------------------------------------------------------------------------------------------------------------

#include "Metal_HLSL.inc"
#include "triangles.h"

#if USE_ROOT_SIGNATURE

// If the shader uses root signatures AND either MTL_FRAMEBUFFER_FETCH_SPACE or MTL_FUNCTION_CONSTANT_SPACE is defined, then
// then you must add MTL_FRAMEBUFFER_FETCH_SPACE- or MTL_FUNCTION_CONSTANT_SPACE-namespaced resources to your root signature where appropriate

#define STR2(x) #x
#define STR(x) STR2(x)

#if defined(MTL_FUNCTION_CONSTANT_SPACE)
#define RootSigFunctionConstantDescTable "DescriptorTable(CBV(b0, space="STR(MTL_FUNCTION_CONSTANT_SPACE)"), CBV(b1, space="STR(MTL_FUNCTION_CONSTANT_SPACE)"), CBV(b2, space="STR(MTL_FUNCTION_CONSTANT_SPACE)")),"
#else
#define RootSigFunctionConstantDescTable
#endif

#if defined(MTL_FRAMEBUFFER_FETCH_SPACE)
#define RootSigFramebufferFetchDescTable "DescriptorTable(SRV(t0, space="STR(MTL_FRAMEBUFFER_FETCH_SPACE)")),"
#else
#define RootSigFramebufferFetchDescTable
#endif

#define RootSig "DescriptorTable(CBV(b0))," RootSigFunctionConstantDescTable RootSigFramebufferFetchDescTable

#endif


cbuffer uniforms : register(b0, space0)
{
    float4 transform1;
    float4 transform2;
    float4 transform3;
};

// Declare vertex function constant
MTL_FUNCTION_CONSTANT(uint, transformIndex, FC_IDX_VERTEX_TRANSFORM_INDEX);

#if USE_ROOT_SIGNATURE
[RootSignature(RootSig)]
#endif
float4 vertexShader(uint vertexID : SV_VertexID, uint instanceID : SV_InstanceID) : SV_Position
{
    float4 position;
    if (vertexID == 0)
    {
        position = float4(-1.0f, -1.0f, 0.0f, 1.0f);
    }
    else if (vertexID == 1)
    {
        position = float4(0.0f, 1.0f, 0.0f, 1.0f);
    }
    else
    {
        position = float4(1.0f, -1.0f, 0.0f, 1.0f);
    }
    
    float4 transform = float4(1.0f, 1.0f, 0.0f, 0.0f);
    uint index = transformIndex;
    if (index == 0)
    {
        transform = transform1;
    }
    else if (index == 1)
    {
        transform = transform2;
    }
    else if (index == 2)
    {
        transform = transform3;
    }
    
    float sx = transform.x;
    float sy = transform.y;
    float tx = transform.z;
    float ty = transform.w;
    
    float3x3 mat = float3x3 (
        sx,   0.0f, tx,
        0.0f, sy,   ty,
        0.0f, 0.0f, 1.0f
    );
    
    float3 result = mul(mat, float3(position.xy, 1));
    return float4(result, 1.0);
}

// Programmable blending using bitwise operations
uint4 logicOp(LOGIC_OP op, uint4 s, uint4 d)
{
    switch (op)
    {
        case LOGIC_OP_CLEAR: return uint4(0,0,0,0);
        case LOGIC_OP_SET: return ~(uint4(0,0,0,0));
        case LOGIC_OP_COPY: return s;
        case LOGIC_OP_COPY_INVERTED: return ~s;
        case LOGIC_OP_NOOP: return d;
        case LOGIC_OP_INVERT: return ~d;
        case LOGIC_OP_AND: return s & d;
        case LOGIC_OP_NAND: return ~(s & d);
        case LOGIC_OP_OR: return s | d;
        case LOGIC_OP_NOR: return ~(s | d);
        case LOGIC_OP_XOR: return s ^ d;
        case LOGIC_OP_EQUIV: return ~(s ^ d);
        case LOGIC_OP_AND_REVERSE: return s & ~d;
        case LOGIC_OP_AND_INVERTED: return ~s & d;
        case LOGIC_OP_OR_REVERSE: return s | ~d;
        case LOGIC_OP_OR_INVERTED: return ~s | d;
        default: break;
    }
    return uint4(0,0,0,0);
}

// Declare fragment function constants
MTL_FUNCTION_CONSTANT(float4, ConstantColor, FC_IDX_FRAGMENT_CONSTANT_COLOR);
MTL_FUNCTION_CONSTANT(int, logicOpMode, FC_IDX_FRAGMENT_LOGIC_OP_MODE);
MTL_FUNCTION_CONSTANT(int, blendState, FC_IDX_FRAGMENT_BLEND_STATE);

#if USE_ROOT_SIGNATURE
[RootSignature(RootSig)]
#endif
half4 pixelShader() : SV_Target0
{
    half4 sourceColor = ConstantColor;
    if (blendState == LogicBlendStateDisabled)
    {
        return sourceColor;
    }
    
    // load pixel color at attachment 0
    half4 destColor = MTL_LOAD_FRAMEBUFFER(0, half4);
    float4 maxValue = float4(0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF);
    
    uint4 sourceColorUint = uint4(float4(sourceColor) * maxValue);
    uint4 destColorUint = uint4(float4(destColor) * maxValue);
    
    // Using logic operations, blend source color from function constant with color at attachment
    LOGIC_OP op = (LOGIC_OP)logicOpMode;
    uint4 finalColor = logicOp(op, sourceColorUint, destColorUint);
    half4 pixelColor = half4(float4(finalColor) / maxValue);
    return pixelColor;
}
